// State
let state = {
  mode: 'addToList', // 'addToList' or 'unfollow'
  handles: [],
  listId: null,
  listName: '',
  added: [],
  failed: [],
  current: null,
  status: 'idle', // idle, running, paused, complete
};

// Elements
const setupView = document.getElementById('setupView');
const runningView = document.getElementById('runningView');
const completeView = document.getElementById('completeView');

const modeTabs = document.getElementById('modeTabs');
const listUrlSection = document.getElementById('listUrlSection');
const handlesInput = document.getElementById('handlesInput');
const loadedCount = document.getElementById('loadedCount');
const listUrlInput = document.getElementById('listUrlInput');
const listStatus = document.getElementById('listStatus');
const startBtn = document.getElementById('startBtn');

const statusBadge = document.getElementById('statusBadge');
const listNameDisplay = document.getElementById('listNameDisplay');
const addedCount = document.getElementById('addedCount');
const failedCount = document.getElementById('failedCount');
const remainingCount = document.getElementById('remainingCount');
const progressFill = document.getElementById('progressFill');
const currentAvatar = document.getElementById('currentAvatar');
const currentHandle = document.getElementById('currentHandle');
const pauseBtn = document.getElementById('pauseBtn');
const stopBtn = document.getElementById('stopBtn');
const log = document.getElementById('log');

const completeStatus = document.getElementById('completeStatus');
const finalAdded = document.getElementById('finalAdded');
const finalFailed = document.getElementById('finalFailed');
const failedSection = document.getElementById('failedSection');
const failedList = document.getElementById('failedList');
const resetBtn = document.getElementById('resetBtn');

// Initialize
document.addEventListener('DOMContentLoaded', async () => {
  // Load state from storage
  const stored = await chrome.storage.local.get(['fixfeed_state']);
  if (stored.fixfeed_state) {
    state = stored.fixfeed_state;
    updateUI();
  }
  
  // Set up mode tabs
  setupModeTabs();
});

// Mode tab handling
function setupModeTabs() {
  modeTabs.querySelectorAll('.mode-tab').forEach(tab => {
    tab.addEventListener('click', () => {
      state.mode = tab.dataset.mode;
      
      // Update tab styles
      modeTabs.querySelectorAll('.mode-tab').forEach(t => t.classList.remove('active'));
      tab.classList.add('active');
      
      // Show/hide list URL section
      if (state.mode === 'unfollow') {
        listUrlSection.style.display = 'none';
        startBtn.textContent = state.handles.length > 0 ? `Unfollow ${state.handles.length} accounts` : 'Start Unfollowing';
        startBtn.classList.add('btn-danger');
      } else {
        listUrlSection.style.display = 'block';
        startBtn.textContent = state.handles.length > 0 && state.listId ? `Add ${state.handles.length} to List` : 'Start Adding';
        startBtn.classList.remove('btn-danger');
      }
      
      checkReady();
      saveState();
    });
  });
}

// Parse handles from textarea input
function parseHandles(text) {
  if (!text.trim()) return [];
  
  return text
    .split(/[\n,]+/) // Split by newline or comma
    .map(h => h.trim())
    .map(h => h.replace(/^@/, '')) // Remove @ prefix
    .filter(h => h.length > 0 && /^[a-zA-Z0-9_]+$/.test(h)) // Valid handle chars
    .filter((h, i, arr) => arr.indexOf(h) === i); // Remove duplicates
}

// Handle textarea input
handlesInput.addEventListener('input', () => {
  state.handles = parseHandles(handlesInput.value);
  loadedCount.textContent = state.handles.length;
  checkReady();
  saveState(); // Save as you type
});

// Also handle paste events
handlesInput.addEventListener('paste', () => {
  // Small delay to let paste complete
  setTimeout(() => {
    state.handles = parseHandles(handlesInput.value);
    loadedCount.textContent = state.handles.length;
    checkReady();
    saveState(); // Save after paste
  }, 10);
});

// Parse list URL to extract ID
function parseListUrl(url) {
  if (!url) return null;
  
  // Match patterns like:
  // https://x.com/i/lists/123456789
  // https://twitter.com/i/lists/123456789
  // x.com/i/lists/123456789
  // Just the ID: 123456789
  
  const match = url.match(/lists\/(\d+)/);
  if (match) {
    return match[1];
  }
  
  // Check if it's just a number
  if (/^\d+$/.test(url.trim())) {
    return url.trim();
  }
  
  return null;
}

// Handle list URL input
listUrlInput.addEventListener('input', () => {
  const url = listUrlInput.value.trim();
  const listId = parseListUrl(url);
  
  if (listId) {
    state.listId = listId;
    state.listName = 'List ' + listId;
    listStatus.textContent = '✓ List ID: ' + listId;
    listStatus.className = 'list-status valid';
  } else if (url) {
    state.listId = null;
    state.listName = '';
    listStatus.textContent = 'Invalid list URL';
    listStatus.className = 'list-status invalid';
  } else {
    state.listId = null;
    state.listName = '';
    listStatus.textContent = '';
    listStatus.className = 'list-status';
  }
  
  checkReady();
  saveState(); // Save as you type
});

// Check if ready to start
function checkReady() {
  let ready = false;
  
  if (state.mode === 'unfollow') {
    ready = state.handles.length > 0;
    if (ready) {
      startBtn.textContent = `Unfollow ${state.handles.length} accounts`;
    } else {
      startBtn.textContent = 'Start Unfollowing';
    }
  } else {
    ready = state.handles.length > 0 && state.listId;
    if (ready) {
      startBtn.textContent = `Add ${state.handles.length} to List`;
    } else {
      startBtn.textContent = 'Start Adding';
    }
  }
  
  startBtn.disabled = !ready;
}

// Start button
startBtn.addEventListener('click', async () => {
  if (state.handles.length === 0) return;
  if (state.mode === 'addToList' && !state.listId) return;
  
  state.status = 'running';
  state.added = [];
  state.failed = [];
  state.current = null;
  
  saveState();
  showView('running');
  listNameDisplay.textContent = state.mode === 'unfollow' ? 'Unfollowing' : state.listName;
  statusBadge.textContent = state.mode === 'unfollow' ? 'Unfollowing...' : 'Running...';
  updateRunningUI();
  
  // Start the process via background script
  chrome.runtime.sendMessage({
    action: 'start',
    mode: state.mode,
    handles: state.handles,
    listId: state.listId,
    listName: state.listName
  });
});

// Pause/Resume
pauseBtn.addEventListener('click', () => {
  if (state.status === 'running') {
    state.status = 'paused';
    pauseBtn.textContent = '▶ Resume';
    statusBadge.textContent = 'Paused';
    statusBadge.className = 'status paused';
    chrome.runtime.sendMessage({ action: 'pause' });
  } else if (state.status === 'paused') {
    state.status = 'running';
    pauseBtn.textContent = '⏸ Pause';
    statusBadge.textContent = 'Running...';
    statusBadge.className = 'status running';
    chrome.runtime.sendMessage({ action: 'resume' });
  }
  saveState();
});

// Stop
stopBtn.addEventListener('click', () => {
  state.status = 'complete';
  chrome.runtime.sendMessage({ action: 'stop' });
  showComplete();
});

// Reset
resetBtn.addEventListener('click', () => {
  state = {
    mode: 'addToList',
    handles: [],
    listId: null,
    listName: '',
    added: [],
    failed: [],
    current: null,
    status: 'idle',
  };
  chrome.storage.local.remove(['fixfeed_state']);
  handlesInput.value = '';
  listUrlInput.value = '';
  listStatus.textContent = '';
  listStatus.className = 'list-status';
  loadedCount.textContent = '0';
  startBtn.disabled = true;
  startBtn.textContent = 'Start Adding';
  startBtn.classList.remove('btn-danger');
  listUrlSection.style.display = 'block';
  modeTabs.querySelectorAll('.mode-tab').forEach(t => t.classList.remove('active'));
  modeTabs.querySelector('[data-mode="addToList"]').classList.add('active');
  showView('setup');
  log.innerHTML = '';
});

// Listen for updates from background script
chrome.runtime.onMessage.addListener((msg) => {
  if (msg.type === 'progress') {
    state.current = msg.handle;
    state.added = msg.added || state.added;
    state.failed = msg.failed || state.failed;
    updateRunningUI();
    addLog(msg.message, msg.success ? 'success' : 'info');
  } else if (msg.type === 'complete') {
    state.status = 'complete';
    state.added = msg.added;
    state.failed = msg.failed;
    saveState();
    showComplete();
  } else if (msg.type === 'error') {
    addLog(msg.message, 'error');
    if (msg.handle) {
      state.failed.push(msg.handle);
    }
    updateRunningUI();
  }
});

// UI Helpers
function showView(view) {
  setupView.classList.toggle('hidden', view !== 'setup');
  runningView.classList.toggle('hidden', view !== 'running');
  completeView.classList.toggle('hidden', view !== 'complete');
}

function updateUI() {
  if (state.handles.length > 0) {
    handlesInput.value = state.handles.map(h => '@' + h).join('\n');
    loadedCount.textContent = state.handles.length;
  }
  
  if (state.listId) {
    listUrlInput.value = 'https://x.com/i/lists/' + state.listId;
    listStatus.textContent = '✓ List ID: ' + state.listId;
    listStatus.className = 'list-status valid';
  }
  
  if (state.status === 'running' || state.status === 'paused') {
    showView('running');
    listNameDisplay.textContent = state.listName;
    updateRunningUI();
    if (state.status === 'paused') {
      pauseBtn.textContent = '▶ Resume';
      statusBadge.textContent = 'Paused';
      statusBadge.className = 'status paused';
    }
  } else if (state.status === 'complete') {
    showComplete();
  } else {
    showView('setup');
    checkReady();
  }
}

function updateRunningUI() {
  const total = state.handles.length;
  const done = state.added.length + state.failed.length;
  const remaining = total - done;
  const percent = total > 0 ? (done / total) * 100 : 0;
  
  addedCount.textContent = state.added.length;
  failedCount.textContent = state.failed.length;
  remainingCount.textContent = remaining;
  progressFill.style.width = percent + '%';
  
  if (state.current) {
    currentHandle.textContent = '@' + state.current;
    currentAvatar.src = 'https://unavatar.io/x/' + state.current;
    currentAvatar.style.display = 'block';
  } else {
    currentHandle.textContent = '—';
    currentAvatar.style.display = 'none';
  }
}

function showComplete() {
  showView('complete');
  finalAdded.textContent = state.added.length;
  finalFailed.textContent = state.failed.length;
  
  // Update label based on mode
  const addedLabel = completeView.querySelector('.stat:first-child .stat-label');
  if (addedLabel) {
    addedLabel.textContent = state.mode === 'unfollow' ? 'UNFOLLOWED' : 'ADDED';
  }
  
  if (state.failed.length > 0) {
    failedSection.classList.remove('hidden');
    failedList.innerHTML = state.failed.map(h => 
      '<a href="https://x.com/' + h + '" target="_blank">@' + h + '</a>'
    ).join('<br>');
  } else {
    failedSection.classList.add('hidden');
  }
  
  if (state.failed.length === 0) {
    completeStatus.textContent = state.mode === 'unfollow' ? '✓ All accounts unfollowed!' : '✓ All accounts added!';
    completeStatus.style.color = '#4ade80';
  } else if (state.added.length === 0) {
    completeStatus.textContent = state.mode === 'unfollow' ? 'Failed to unfollow accounts' : 'Failed to add accounts';
    completeStatus.style.color = '#f87171';
  } else {
    const action = state.mode === 'unfollow' ? 'Unfollowed' : 'Added';
    completeStatus.textContent = `${action} ${state.added.length}, ${state.failed.length} failed`;
    completeStatus.style.color = '#fbbf24';
  }
}

function addLog(message, type = 'info') {
  const entry = document.createElement('div');
  entry.className = 'log-entry ' + type;
  entry.textContent = message;
  log.insertBefore(entry, log.firstChild);
  
  // Keep only last 50 entries
  while (log.children.length > 50) {
    log.removeChild(log.lastChild);
  }
}

function saveState() {
  chrome.storage.local.set({ fixfeed_state: state });
}
