// Content script - runs on x.com/twitter.com
// Handles adding accounts to lists via X's internal API

console.log('FixYourFeed content script loaded');

// Listen for messages from popup/background
chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  if (msg.action === 'ping') {
    sendResponse({ ok: true });
    return true;
  }
  
  if (msg.action === 'addToList') {
    addAccountToList(msg.handle, msg.listId)
      .then(result => sendResponse(result))
      .catch(err => sendResponse({ success: false, error: err.message }));
    return true; // Keep channel open for async
  }
  
  if (msg.action === 'unfollow') {
    unfollowAccount(msg.handle)
      .then(result => sendResponse(result))
      .catch(err => sendResponse({ success: false, error: err.message }));
    return true;
  }
  
  if (msg.action === 'resolveUserId') {
    resolveUserIdToHandle(msg.userId)
      .then(result => sendResponse(result))
      .catch(err => sendResponse({ success: false, error: err.message }));
    return true;
  }
});

// Listen for messages from web page (for resolve handles feature)
window.addEventListener('message', async (event) => {
  if (event.data && event.data.type === 'FIXFEED_RESOLVE') {
    const result = await resolveUserIdToHandle(event.data.userId);
    window.postMessage({
      type: 'FIXFEED_RESOLVED',
      userId: event.data.userId,
      ...result
    }, '*');
  }
});

// Get CSRF token from cookie
function getCsrfToken() {
  const match = document.cookie.match(/ct0=([^;]+)/);
  return match ? match[1] : null;
}

// Get bearer token - X's public bearer token used by their web app
function getBearerToken() {
  return 'AAAAAAAAAAAAAAAAAAAAANRILgAAAAAAnNwIzUejRCOuH5E6I8xnZz4puTs%3D1Zv7ttfk8LF81IUq16cHjhLTvJu4FA33AGWWjCpTnA';
}

// Make authenticated request to X API
async function xApiRequest(url, options = {}) {
  const csrfToken = getCsrfToken();
  if (!csrfToken) {
    throw new Error('Not logged in to X');
  }
  
  const headers = {
    'authorization': `Bearer ${getBearerToken()}`,
    'x-csrf-token': csrfToken,
    'x-twitter-auth-type': 'OAuth2Session',
    'x-twitter-active-user': 'yes',
    'x-twitter-client-language': 'en',
    ...options.headers,
  };
  
  const response = await fetch(url, {
    ...options,
    headers,
    credentials: 'include',
  });
  
  if (!response.ok) {
    const text = await response.text();
    console.error('X API error:', response.status, text);
    throw new Error(`API error: ${response.status}`);
  }
  
  return response.json();
}

// Get user ID from screen name using X's REST API (more reliable)
async function getUserId(screenName) {
  console.log('Looking up user:', screenName);
  
  // Use the simpler REST-style endpoint
  const url = `https://x.com/i/api/1.1/users/show.json?screen_name=${encodeURIComponent(screenName)}`;
  
  try {
    const data = await xApiRequest(url);
    const userId = data?.id_str;
    
    if (!userId) {
      console.error('User lookup response:', data);
      throw new Error('User not found');
    }
    
    console.log('Found user ID:', userId, 'for', screenName);
    return userId;
  } catch (e) {
    console.error('Error looking up user:', screenName, e);
    
    // Fallback: try GraphQL endpoint
    return getUserIdGraphQL(screenName);
  }
}

// Fallback: Get user ID via GraphQL
async function getUserIdGraphQL(screenName) {
  console.log('Trying GraphQL lookup for:', screenName);
  
  const variables = {
    screen_name: screenName,
    withSafetyModeUserFields: true,
  };
  
  const features = {
    hidden_profile_subscriptions_enabled: true,
    rweb_tipjar_consumption_enabled: true,
    responsive_web_graphql_exclude_directive_enabled: true,
    verified_phone_label_enabled: false,
    subscriptions_verification_info_is_identity_verified_enabled: true,
    subscriptions_verification_info_verified_since_enabled: true,
    highlights_tweets_tab_ui_enabled: true,
    responsive_web_twitter_article_notes_tab_enabled: true,
    subscriptions_feature_can_gift_premium: true,
    creator_subscriptions_tweet_preview_api_enabled: true,
    responsive_web_graphql_skip_user_profile_image_extensions_enabled: false,
    responsive_web_graphql_timeline_navigation_enabled: true,
  };
  
  const fieldToggles = {
    withAuxiliaryUserLabels: false,
  };
  
  const url = `https://x.com/i/api/graphql/BQ6xjFU6Mgm-WhEP3OiT9w/UserByScreenName?variables=${encodeURIComponent(JSON.stringify(variables))}&features=${encodeURIComponent(JSON.stringify(features))}&fieldToggles=${encodeURIComponent(JSON.stringify(fieldToggles))}`;
  
  const data = await xApiRequest(url);
  const userId = data?.data?.user?.result?.rest_id;
  
  if (!userId) {
    throw new Error('User not found');
  }
  
  console.log('Found user ID via GraphQL:', userId);
  return userId;
}

// Add user to list using REST API
async function addUserToList(userId, listId) {
  console.log('Adding user', userId, 'to list', listId);
  
  // Try REST API first (more reliable)
  const url = 'https://x.com/i/api/1.1/lists/members/create.json';
  
  const formData = new URLSearchParams();
  formData.append('list_id', listId);
  formData.append('user_id', userId);
  
  try {
    const data = await xApiRequest(url, {
      method: 'POST',
      headers: {
        'content-type': 'application/x-www-form-urlencoded',
      },
      body: formData.toString(),
    });
    
    console.log('Successfully added user to list via REST');
    return true;
  } catch (e) {
    console.log('REST API failed, trying GraphQL...', e);
    return addUserToListGraphQL(userId, listId);
  }
}

// Fallback: Add user to list via GraphQL
async function addUserToListGraphQL(userId, listId) {
  console.log('Trying GraphQL add for user', userId);
  
  const url = 'https://x.com/i/api/graphql/lLNsL7mW6gSEQG6rXP7TNw/ListAddMember';
  
  const body = {
    variables: {
      listId: listId,
      userId: userId,
    },
    features: {
      rweb_lists_timeline_redesign_enabled: true,
      responsive_web_graphql_exclude_directive_enabled: true,
      verified_phone_label_enabled: false,
      responsive_web_graphql_skip_user_profile_image_extensions_enabled: false,
      responsive_web_graphql_timeline_navigation_enabled: true,
    },
    queryId: 'lLNsL7mW6gSEQG6rXP7TNw',
  };
  
  const data = await xApiRequest(url, {
    method: 'POST',
    headers: {
      'content-type': 'application/json',
    },
    body: JSON.stringify(body),
  });
  
  if (data.errors && data.errors.length > 0) {
    throw new Error(data.errors[0].message || 'Failed to add');
  }
  
  console.log('Successfully added user to list via GraphQL');
  return true;
}

// Main function to add account to list
async function addAccountToList(handle, listId) {
  console.log(`Adding @${handle} to list ${listId}`);
  
  try {
    // Step 1: Get user ID from handle
    const userId = await getUserId(handle);
    
    // Step 2: Add user to list  
    await addUserToList(userId, listId);
    
    return { success: true };
  } catch (e) {
    console.error(`Failed to add @${handle}:`, e);
    return { success: false, error: e.message };
  }
}

// Unfollow account - accepts handle, user_id, or profile URL
async function unfollowAccount(input) {
  console.log(`Unfollowing:`, input);
  
  try {
    var userId;
    
    // Check if input is a user ID URL
    if (input.includes('user_id=')) {
      userId = input.match(/user_id=(\d+)/)?.[1];
      console.log('Extracted user ID from URL:', userId);
    } 
    // Check if input is a direct user ID (just numbers)
    else if (/^\d+$/.test(input)) {
      userId = input;
      console.log('Input is direct user ID:', userId);
    }
    // Otherwise it's a handle - need to look up user ID
    else {
      var handle = input.replace(/^@/, '').replace(/https?:\/\/(x|twitter)\.com\//, '');
      userId = await getUserId(handle);
    }
    
    if (!userId) {
      throw new Error('Could not determine user ID');
    }
    
    // Unfollow using REST API
    const url = 'https://x.com/i/api/1.1/friendships/destroy.json';
    
    const formData = new URLSearchParams();
    formData.append('user_id', userId);
    
    try {
      await xApiRequest(url, {
        method: 'POST',
        headers: {
          'content-type': 'application/x-www-form-urlencoded',
        },
        body: formData.toString(),
      });
      
      console.log('Successfully unfollowed via REST');
      return { success: true };
    } catch (e) {
      console.log('REST API failed, trying GraphQL...', e);
      return unfollowGraphQL(userId);
    }
  } catch (e) {
    console.error(`Failed to unfollow:`, input, e);
    return { success: false, error: e.message };
  }
}

// Fallback: Unfollow via GraphQL
async function unfollowGraphQL(userId) {
  console.log('Trying GraphQL unfollow for user', userId);
  
  const url = 'https://x.com/i/api/graphql/QY4RKCLEzgkwvtUd3x_n2Q/DestroyFriendship';
  
  const body = {
    variables: {
      target_user_id: userId,
    },
    queryId: 'QY4RKCLEzgkwvtUd3x_n2Q',
  };
  
  const data = await xApiRequest(url, {
    method: 'POST',
    headers: {
      'content-type': 'application/json',
    },
    body: JSON.stringify(body),
  });
  
  if (data.errors && data.errors.length > 0) {
    throw new Error(data.errors[0].message || 'Failed to unfollow');
  }
  
  console.log('Successfully unfollowed via GraphQL');
  return { success: true };
}

// Resolve user ID to handle by looking up user info
async function resolveUserIdToHandle(userId) {
  console.log('Resolving user ID to handle:', userId);
  
  try {
    // Use the users/show endpoint with user_id
    const url = `https://x.com/i/api/1.1/users/show.json?user_id=${userId}`;
    
    const data = await xApiRequest(url);
    const handle = data?.screen_name;
    
    if (!handle) {
      console.error('No handle in response:', data);
      return { success: false, error: 'No handle found' };
    }
    
    console.log('Resolved', userId, 'to @' + handle);
    return { success: true, handle: handle.toLowerCase() };
  } catch (e) {
    console.error('Error resolving user ID:', userId, e);
    
    // Try GraphQL fallback
    try {
      return await resolveUserIdGraphQL(userId);
    } catch (e2) {
      return { success: false, error: e2.message };
    }
  }
}

// Fallback: Resolve via GraphQL
async function resolveUserIdGraphQL(userId) {
  console.log('Trying GraphQL resolve for:', userId);
  
  const variables = {
    userId: userId,
    withSafetyModeUserFields: true,
  };
  
  const features = {
    hidden_profile_subscriptions_enabled: true,
    responsive_web_graphql_exclude_directive_enabled: true,
    verified_phone_label_enabled: false,
    responsive_web_graphql_skip_user_profile_image_extensions_enabled: false,
    responsive_web_graphql_timeline_navigation_enabled: true,
  };
  
  const url = `https://x.com/i/api/graphql/xf3jd90KKBCUxdlI_tNHZw/UserByRestId?variables=${encodeURIComponent(JSON.stringify(variables))}&features=${encodeURIComponent(JSON.stringify(features))}`;
  
  const data = await xApiRequest(url);
  const handle = data?.data?.user?.result?.legacy?.screen_name;
  
  if (!handle) {
    throw new Error('User not found');
  }
  
  console.log('Resolved via GraphQL:', userId, 'to @' + handle);
  return { success: true, handle: handle.toLowerCase() };
}
