// Background service worker
// Coordinates the add-to-list, unfollow, and resolve handles processes

let state = {
  running: false,
  paused: false,
  mode: 'addToList',
  handles: [],
  listId: null,
  listName: '',
  added: [],
  failed: [],
  currentIndex: 0,
};

// Listen for messages from popup and web pages
chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  if (msg.action === 'start') {
    startProcess(msg.mode, msg.handles, msg.listId, msg.listName);
  } else if (msg.action === 'pause') {
    state.paused = true;
  } else if (msg.action === 'resume') {
    state.paused = false;
    processNext();
  } else if (msg.action === 'stop') {
    state.running = false;
    state.paused = false;
  } else if (msg.action === 'resolveUserId') {
    // Handle resolve request - forward to X tab
    resolveUserIdViaTab(msg.userId).then(sendResponse);
    return true; // Keep channel open for async
  }
});

// Listen for external messages (from fixyourfeed.xyz)
chrome.runtime.onMessageExternal.addListener((msg, sender, sendResponse) => {
  if (msg.action === 'resolveUserId') {
    resolveUserIdViaTab(msg.userId).then(sendResponse);
    return true;
  }
});

// Resolve user ID by sending request to X tab
async function resolveUserIdViaTab(userId) {
  try {
    let tab = await findXTab();
    if (!tab) {
      // Open X in background
      tab = await chrome.tabs.create({ url: 'https://x.com', active: false });
      await new Promise(resolve => setTimeout(resolve, 3000));
    }
    
    const injected = await ensureContentScript(tab.id);
    if (!injected) {
      return { success: false, error: 'Could not connect to X' };
    }
    
    const response = await chrome.tabs.sendMessage(tab.id, {
      action: 'resolveUserId',
      userId: userId
    });
    
    return response;
  } catch (e) {
    console.error('Error resolving user ID:', e);
    return { success: false, error: e.message };
  }
}

// Ensure content script is injected
async function ensureContentScript(tabId) {
  try {
    // Try to ping the content script
    await chrome.tabs.sendMessage(tabId, { action: 'ping' });
    return true;
  } catch (e) {
    // Content script not loaded, inject it
    console.log('Injecting content script into tab', tabId);
    try {
      await chrome.scripting.executeScript({
        target: { tabId: tabId },
        files: ['content.js']
      });
      // Wait a bit for it to initialize
      await new Promise(resolve => setTimeout(resolve, 500));
      return true;
    } catch (err) {
      console.error('Failed to inject content script:', err);
      return false;
    }
  }
}

// Start the process
async function startProcess(mode, handles, listId, listName) {
  state = {
    running: true,
    paused: false,
    mode: mode || 'addToList',
    handles: handles,
    listId: listId,
    listName: listName,
    added: [],
    failed: [],
    currentIndex: 0,
  };
  
  // Save state
  await chrome.storage.local.set({ fixfeed_process: state });
  
  // Find or create a tab on x.com
  let tab = await findXTab();
  if (!tab) {
    tab = await chrome.tabs.create({ url: 'https://x.com', active: false });
    // Wait for tab to load
    await new Promise(resolve => setTimeout(resolve, 3000));
  }
  
  state.tabId = tab.id;
  
  // Ensure content script is injected
  const injected = await ensureContentScript(tab.id);
  if (!injected) {
    notifyPopup({
      type: 'error',
      message: 'Could not connect to X tab. Please refresh x.com and try again.',
    });
    return;
  }
  
  // Start processing
  processNext();
}

// Find existing X tab
async function findXTab() {
  const tabs = await chrome.tabs.query({ url: ['https://x.com/*', 'https://twitter.com/*'] });
  return tabs[0] || null;
}

// Process next account
async function processNext() {
  if (!state.running || state.paused) return;
  
  if (state.currentIndex >= state.handles.length) {
    // Complete
    state.running = false;
    notifyPopup({
      type: 'complete',
      added: state.added,
      failed: state.failed,
    });
    return;
  }
  
  const handle = state.handles[state.currentIndex];
  const actionVerb = state.mode === 'unfollow' ? 'Unfollowing' : 'Processing';
  
  notifyPopup({
    type: 'progress',
    handle: handle,
    added: state.added,
    failed: state.failed,
    message: `${actionVerb} @${handle}...`,
    success: false,
  });
  
  try {
    // Ensure content script is still there
    await ensureContentScript(state.tabId);
    
    // Send message to content script
    const action = state.mode === 'unfollow' ? 'unfollow' : 'addToList';
    const response = await chrome.tabs.sendMessage(state.tabId, {
      action: action,
      handle: handle,
      listId: state.listId,
    });
    
    if (response && response.success) {
      state.added.push(handle);
      const successVerb = state.mode === 'unfollow' ? 'Unfollowed' : 'Added';
      notifyPopup({
        type: 'progress',
        handle: handle,
        added: state.added,
        failed: state.failed,
        message: `✓ ${successVerb} @${handle}`,
        success: true,
      });
    } else {
      throw new Error(response?.error || 'Unknown error');
    }
  } catch (e) {
    console.error(`Failed to process @${handle}:`, e);
    state.failed.push(handle);
    notifyPopup({
      type: 'error',
      handle: handle,
      message: `✗ Failed @${handle}: ${e.message}`,
    });
  }
  
  state.currentIndex++;
  
  // Save progress
  await chrome.storage.local.set({ fixfeed_process: state });
  
  // Wait before next (rate limiting)
  const delay = 2000 + Math.random() * 1000; // 2-3 seconds
  setTimeout(processNext, delay);
}

// Notify popup of updates
function notifyPopup(msg) {
  chrome.runtime.sendMessage(msg).catch(() => {
    // Popup might be closed, that's ok
  });
}

// Restore state on startup
chrome.runtime.onStartup.addListener(async () => {
  const stored = await chrome.storage.local.get(['fixfeed_process']);
  if (stored.fixfeed_process && stored.fixfeed_process.running) {
    state = stored.fixfeed_process;
    // Resume if was running
    if (!state.paused) {
      const tab = await findXTab();
      if (tab) {
        state.tabId = tab.id;
        processNext();
      }
    }
  }
});
